package gov.va.med.mhv.usermgmt.persist.ldap;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import netscape.ldap.LDAPCache;
import netscape.ldap.LDAPException;
import netscape.ldap.LDAPv3;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.util.StringUtils;

/**
 * This class models a set of update operations against a single LDAP distinguished name (dn).
 * These commands can add, replace, and remove attributes and their values as well as operate
 * on entries themselves.
 * <p>
 * A typical usage of this class would be something along the lines of:
 * <p>
 * <code>
 * 		// get a username from somewhere<br>
 * 		String userName = "joeUser";<br> 
 * 		LdapUpdate update = new LdapUpdate(LdapUtil.buildDistinguishedName(userName));<br>
 * 		// schedule one or more commands to execute, where "userPassword" is the name<br>
 *		// of the attribute to be modified<br>
 *		String targetAttributeName = "userPassword";<br>
 * 		update.addCommand(LdapAttributeModification.wipeAndReplace(targetAttributeName, "someNewPassword"));<br>
 * 		// execute the scheduled commands<br>
 * 		update.executeUpdate();<br>
 * </code>
 * @author Jon Crater Apr 4, 2006 3:21:02 PM
 * @see gov.va.med.mhv.user.persist.ldap.LdapFunction
 * @see gov.va.med.mhv.user.persist.ldap.LdapUtil#buildUserDn(String);
 */
public class LdapUpdate extends LdapFunction {
	private Log logger = LogFactory.getLog(getClass());
	
	/**
	 * The distinguished name of the LDAP entry to which the commands contained by this update
	 * instance will be applied.
	 */
	private String dn;
	
	/**
	 * The list of <code>LdapCommand</code> instances which will be executed when 
	 * executeUpdate() is invoked.
	 */
	private List commands = new ArrayList(4);

	public LdapUpdate(String dn) {
		setDn(dn);
	}
	
	/**
	 * Iterate through the configured commands scheduled for update, executing
	 * each in the order it was added via the <code>addCommand(LdapCommand)</code> method.
	 * @return void
	 */
	public void executeUpdate() {
		validate();

		// iterate the configured commands and execute each
		LdapCommand command = null;
		try {
			for(Iterator i = commands.iterator(); i.hasNext();) {
				command = (LdapCommand)i.next();
				command.execute(getDn(), conn);
			}
		}
		catch(LDAPException e) {
			logger.error("Error performing update for dn '" + getDn() + "'");
			throw getExceptionTranslator().translate(e);
			/*throw new LdapException(
				"Error executing LdapCommand [" + command.getClass().getName() + "]", e, getDn());*/
		}
		finally {
			resetCache(dn);
			disconnect();
		}
	}
	
	/**
	 * Schedule an <code>LdapCommand</code> instance to be executed.
	 * @param command
	 */
	public void addCommand(LdapCommand command) {
		if(command == null)
			throw new NullPointerException();
		
		commands.add(command);
	}

	/**
	 * Set the distinguished name.
	 * @param dn
	 */
	public void setDn(String dn) {
		if(dn == null)
			throw new NullPointerException("You must supply a distinguished name which this update will affect.");

		this.dn = dn;
	}
	
	/**
	 * Clear all scheduled LdapCommand instances.
	 */
	public void clearCommands() {
		this.commands.clear();
	}

	/**
	 * Get the distinguished name.
	 * @return String
	 */
	protected String getDn() {
		return dn;
	}

	/**
	 * Reset the connection level cache for the associated distinguished name.
	 * @param dn The distinguished name for which the cache will be reset.
	 */
	protected void resetCache(String dn) {
		LDAPCache ldapcache = conn.getCache();
		if(ldapcache != null)
			ldapcache.flushEntries(dn, LDAPv3.SCOPE_BASE);
	}
	
	protected void validate() {
		if(!conn.isConnected())
			throw new IllegalStateException("LDAPConnection is not connected to the server.");
		
		if(commands.isEmpty())
			throw new IllegalStateException("No LdapCommand instances are configured.");
		
		if(!StringUtils.hasText(getDn()))
			throw new IllegalStateException("No distinguishedName configured.");
	}
}